import {
  Component,
  OnInit,
  OnDestroy,
  Input,
  ViewChild,
  Renderer
} from '@angular/core'
import { TimerObservable } from "rxjs/observable/TimerObservable"
import { Config, ConfigService } from '../../core/config/config.service'
import { LoggerService } from '../../core/logger/logger.service'
import { AppCommService } from '../../core/app-comm/app-comm.service'

import { Observable, Subscription } from 'rxjs';

@Component({
  selector: 'app-link',
  templateUrl: './link.component.html',
  styleUrls: ['./link.component.css']
})
export class LinkComponent implements OnInit, OnDestroy {
  secondsConfigured: number
  seconds: number
  localDomains: Array<string>
  @ViewChild('linkGo') linkGo: any

  get plural(): string {
    return this.seconds === 1 ? "" : "s"
  }

  @Input() href: string

  @Input() text: string

  clickTimer: Observable<number>
  clickTimerSubscription: Subscription
  linkOpen: Subscription

  constructor(private configService: ConfigService,
    private loggerService: LoggerService,
    private renderer: Renderer,
    private appCommService: AppCommService) {
    this.loggerService.info('LinkComponent::info')
    this.loggerService.warn('LinkComponent::warn')
    // this.loggerService.error('LinkComponent::error')
    this.loggerService.debug('LinkComponent::constructor')

  }

  click(): void {
    this.loggerService.debug('LinkComponent::click')

    this.appCommService.linkNoticeOpen(this.href);

    this.linkOpen = this.appCommService.linkNotice$.subscribe(
          linkId => {
            this.cancel()
        });

    const localMatches = this.localDomains.filter(p => this.href.startsWith(p))
    if (localMatches.length > 0 || this.secondsConfigured === 0) {
      this.go()
      return
    }
    if (typeof this.clickTimerSubscription !== 'undefined' && typeof this.clickTimerSubscription.unsubscribe === 'function') {
      console.log(`Clearing LinkComponent observable`)
      this.clickTimerSubscription.unsubscribe()
    }
    this.clickTimer = TimerObservable.create(1000, 1000)
    this.seconds = this.secondsConfigured
    this.clickTimerSubscription = this.clickTimer.subscribe((t: any) => {
      this.seconds--
      if (this.seconds < 1) {
        this.killTimer()
        this.go()
      }
    })

    setTimeout(() => {
      this.renderer.invokeElementMethod(this.linkGo.nativeElement, 'focus')
    }, 100)

  }

  killTimer() {
    try {
      this.seconds = 0
      this.loggerService.debug('LinkComponent::killTimer')
      if (typeof this.clickTimerSubscription !== 'undefined' && typeof this.clickTimerSubscription.unsubscribe === 'function') {
        this.clickTimerSubscription.unsubscribe()
      }
    } catch (ex) {
      this.loggerService.debug(`ex:${JSON.stringify(ex)}`)
    }
  }

  cancel(): void {
    this.linkOpen.unsubscribe()
    this.killTimer()
  }

  go(): void {
    this.loggerService.debug('LinkComponent::go')
    window.location.href = this.href
  }

  async ngOnInit() {
    this.loggerService.debug('LinkComponent::ngOnInit')
    this.secondsConfigured = await this.configService.load(Config.SecondsForExternalRedirect)
    this.localDomains = await this.configService.load(Config.LocalDomains)
  }

  ngOnDestroy() {
    this.loggerService.debug('LinkComponent::ngOnDestroy')
    this.killTimer()
  }
}
